/*
 Arduino UNO DDS-60/AD9850 WSPR/QRSS Controller

 Generates QRSS and WSPR coordinated frequency hopping transmissions 
 on 6 thru 160 meters synchronized by either GPS or WWVB time data.
 
 This sketch is configured for DDS60 operation. It may be re-configured
 for AD9850 operation by changing W0 to 0 and setting fCLK to the AD9850 
 clock frequency.    
 
 Acknowlegements    
 The on-chip generation of the WSPR message algorithm is the work of 
 Andy Talbot, G4JNT. The EEPROM read/write routine is courtesy of of
 G0MGX. Portions of the GPS receive code were influenced by 
 Igor Gonzalez Martin's Arduino tutorial.
 
 Copyright (C) 2011,  Gene Marcus W3PM GM4YRE
 
 16 May 2013  Modified to compile with Arduino IDE 1.04 
              Corrected satellite display data bug
 
 Permission is granted to use, copy, modify, and distribute this software
 and documentation for non-commercial purposes.

 Harware: Arduino Nano (Old Bootloader)

 Modifications made by Rob PA0RWE december 2015  (v3_0)
 -  Because DCF was not stable enough I switched to GPS
 -  Removed QRSS and WWVB stuff
 -  Found that I2C (for the display) and Timer1 isn't a good combi.

 Modifications made by Rob PA0RWE march 2021  (v4_0)
 -  Added choice of AD9850 or AD9851 by w0 variable at line 122

 Modifications made by Rob PA0RWE januari 2022  (v4_1)
 -  GPS Output to RX0 (removable connection for programming) 
 -  Frequency of 80m has been changed
 -  Added DDS reset procedure for secure start-up
 -  Added intro screen
 -  Cosmetic improvements

 Modifications made by Rob PA0RWE november 2023  (v4_2)
 -  Decreased calibration steps to 100 Hz

 Major Modifications made by Rob PA0RWE february 2024  (v5_0)
 -  Cleaning the software, trying to prevent display crash...
 -  Add the possibility to do transmitting on 1 frequency each 6 minutes with the FIX.FREQ button
    and select a timeslot with the + or - button as well..
 -  Add or subtract steps of 25Hz to the transmitting frequency with CAL/OFFSET button
 -  Add active Band information (e.g. 80m)

 Modifications made by Rob PA0RWE november 2025  (v5_1)
 -  Added band selection output on pin D9-D11

 _________________________________________________________________________
 
  ***** REMOVE PIN RX0 CONNECTION BEFORE PROGRAMMING *****
 _________________________________________________________________________
 
 UNO Digital Pin Allocation
 D0  TX0
 D1  RX0 GPS input (TX)
 D2  PB1 CAL / OFFSET
 D3  PB2 TX fixed freq (irq)
 D4  load
 D5  clock
 D6  data
 D7  PB4 Cal/Offset/Slot -
 D8  PB3 Cal/Offset/Slot +
 D9  Band (1)
 D10 Band (2)
 D11 Band (4)
 D12 GPS Pulse (LED)
 D13 TX (LED)
 A0/D14 LCD D7
 A1/D15 LCD D6
 A2/D16 LCD D5
 A3/D17 LCD D4
 A4/D18 LCD enable
 A5/D19 LCD RS 
 ------------------------------------------------------------
 */
// include the library code:
#include <LiquidCrystal.h>
#include <EEPROM.h>
#include <MsTimer2.h>

struct CalFact {
  union {
    long value;
    struct {
      unsigned char b1;
      unsigned char b2;
      unsigned char b3;
      unsigned char b4;
    }
    __attribute__((packed));
  }
  __attribute__((packed));
}
__attribute__((packed));
struct CalFact CalFactor;


//__________________________________________________________________________________________________
// ENTER WSPR DATA:
char call[7] = "xxxxxx";    // Your call
char locator[5] = "xxxx";   // Use 4 character locator e.g. "EM64"
byte power = 13;            // Min = 0 dBm, Max = 43 dBm, steps 0,3,7,10,13,17,20,23,27,30,33,37,40,43
//__________________________________________________________________________________________________

//__________________________________________________________________________________________________
// LOAD BAND FREQUENCY DATA:  Equal to WSPR bandhopping and at the START of every band
unsigned long band[10] ={
  1838000,  // timeslot 0  00 / 20 / 40 minutes after hour    Band 1
  3570000,  // timeslot 1  02 / 22 / 42  minutes after hour   Band 2
  50294400, // timeslot 2  04 / 24 / 44  minutes after hour   Band 3 (Replacement for 60m to stay in sync with Band hopping)
  7040000,  // timeslot 3  06 / 26 / 46  minutes after hour   Band 4
  10140100, // timeslot 4  08 / 28 / 48  minutes after hour   Band 5
  14097000, // timeslot 5  10 / 30 / 50  minutes after hour   Band 6
  18106000, // timeslot 6  12 / 32 / 52  minutes after hour   Band 6
  21096000, // timeslot 7  14 / 34 / 54  minutes after hour   Band 7
  24926000, // timeslot 8  16 / 36 / 56  minutes after hour   Band 7
  28126000  // timeslot 9  18 / 38 / 58  minutes after hour   Band 7
};

// Load BandSelect data for band filtering
unsigned int bandSelect[10] ={1,2,3,4,5,6,6,7,7,7};

//__________________________________________________________________________________________________
// LOAD TRANSMIT TIME SLOT DATA: ( 0=idle, 1=transmit WSPR)
// Note: Timeslot is depending on minute time
const byte TransmitFlag[10] ={
  1, // timeslot 0
  1, // timeslot 1
  1, // timeslot 2 
  1, // timeslot 3
  1, // timeslot 4
  1, // timeslot 5 
  1, // timeslot 6
  1, // timeslot 7
  1, // timeslot 8
  1  // timeslot 9
};


// AD9850 AD9851 selection 
byte w0 = 1;            // 0 = AD9850, 1 = AD9851 (DDS60)
unsigned long fCLK;     // 125 MHz: AD9850, 180 MHz: AD9851               


//__________________________________________________________________________________________________

const char SyncVec[162] = {
  1,1,0,0,0,0,0,0,1,0,0,0,1,1,1,0,0,0,1,0,0,1,0,1,1,1,1,0,0,0,0,0,0,0,1,0,0,1,0,1,0,0,0,0,0,0,1,0,
  1,1,0,0,1,1,0,1,0,0,0,1,1,0,1,0,0,0,0,1,1,0,1,0,1,0,1,0,1,0,0,1,0,0,1,0,1,1,0,0,0,1,1,0,1,0,1,0,
  0,0,1,0,0,0,0,0,1,0,0,1,0,0,1,1,1,0,1,1,0,0,1,1,0,1,0,0,0,1,1,1,0,0,0,0,0,1,0,1,0,0,1,1,0,0,0,0,
  0,0,0,1,1,0,1,0,1,1,0,0,0,1,1,0,0,0
};

// Calibrate / Fixfreq / Offset buttons
#define FixFreqButton     3         // IRQ
#define CalOffsetButton   2
#define CalOffUpButton    8
#define CalOffDwnButton   7

// NMEA fields
#define RMC_TIME 1
#define GGA_FIX 6
#define GGA_SATS 7
#define HEADER_RMC "$GPRMC"
#define HEADER_GGA "$GPGGA"

// initialize the library with the numbers of the interface pins
LiquidCrystal lcd(19,18,17,16,15,14);


// configure variables
int txPin = 13;                     // TX (HIGH on transmit)
int txLed = 12;                     // LED for Tx monitoring
const int sentenceSize = 82;
const int max_nr_words = 20;
char sentence[sentenceSize], sz[20];
char* words [max_nr_words];

volatile int second=0, minute=0, hour=0;
float time;
int gps_sats;
boolean dispok=false;
int mSecTimer2;

// DDS Control
byte data  = 6;                     //DATA 
byte clock = 5;                     //CLOCK 
byte load  = 4;                     //LOAD 
byte count = 0;
unsigned long FreqWord, TempWord;
unsigned long TenMHz = 10000000;    // Calibration freq
int BandOffset = 0;
unsigned int OffsetFreq[4];

volatile byte bb, i, j, ii, timeslot=0, fixedslot = 0;
byte symbol[162];
byte c[11];                         // encoded message
byte sym[170];                      // symbol table 162
byte symt[170];                     // symbol table temp
byte RXflag=1;                      // GPS receiver control 0 = disable, 1 = enable
volatile byte FixFreqFlag = 0, GPSinhibitFlag = 0, NoDispFlag = 0, FixonFlag = 0, TxOnFlag = 0;
unsigned long n1;                   // encoded callsign
unsigned long m1;                   // encodes locator

const char Band[10][4] = {"160", "80m", " 6m", "40m", "30m", "20m", "17m", "15m", "12m", "10m"};

//******************************************************************
// Clock - interrupt routine used as master timekeeper
// Timer1 Overflow Interrupt Vector, called every second
//******************************************************************
ISR(TIMER1_COMPA_vect) {
  second++ ;
  if (second == 60) {
    minute++ ;
    second=0 ;
  }
  if (minute == 60) {
    hour++;
    minute=0 ;
  }
  if (hour == 24) {
    hour=0 ;
  }

  if (NoDispFlag == 0) displaytime();           // No display during Offset or FixedSLot setting

  if((bitRead(minute,0) == 0) & (second == 0))     
  {
    if (minute < 20) {
      timeslot = minute/2;
    }
    else{
      if (minute < 40) {
        timeslot = (minute-20)/2;
      }
      else {
        timeslot = (minute -40)/2;
      }
    }
    if (FixonFlag)              // Fixed Timeslot is active
    {
      timeslot = fixedslot;
    }
  setfreq(); 
  if (TxOnFlag) transmit();  // Transmit depending on FixSlot
  }  
}

//******************************************************************
//Timer2 Overflow Interrupt Vector, called every mSec to increment
//the mSecTimer2 used for WSPR transmit timing.
//******************************************************************
ISR(TIMER2_COMPA_vect) {
// WSPR symbol tranmission routine begins here:
  mSecTimer2++;
  if(mSecTimer2 > 681){
    mSecTimer2 = 0;
    if(bb < 3) {                    // Begin 2 second delay - actually 0.682mSec * 3
      TempWord = FreqWord;
      TransmitSymbol();
      bb++;
    }
    else
    {
    if (count < 162)                // Begin 162 WSPR symbol transmission
      {
        TempWord = FreqWord + OffsetFreq[sym[count]];
        TransmitSymbol();
        count++;                    //Increments the interrupt counter
      }
    else
      {
        TIMSK2 = 0;                 // Disable WSPR timer
        digitalWrite(txPin,LOW);    // External transmit control OFF 
        digitalWrite(txLed,LOW);    // Tx Led OFF
        TempWord=0;                 // Turn off transmitter
        TransmitSymbol(); 
        RXflag = 1;                 // Turn GPS receiver back on 
        lcd.setCursor(8,1);      
        lcd.print(F(" IDLE   "));
      }
    }
  }
}


/******************************************************************
 * S E T U P 
 ******************************************************************/
void setup()
{
  //Set up Timer2 to fire every mSec (WSPR timer)
  TIMSK2 = 0;        // Disable timer during setup
  TCCR2A = 2;        // CTC mode
  TCCR2B = 4;        // Timer Prescaler set to 64
  OCR2A = 247;       // Timer set for 1 mSec with correction factor

  //Set up Timer1A to fire every second (master clock)
  TCCR1B = 0;        // Disable timer during setup
  TIMSK1 = 2;        // Timer1 Interrupt enable
  TCCR1A = 0;        // Normal port operation, Wave Gen Mode normal
  TCCR1B = 12;       // Timer prescaler to 256 - CTC mode
  OCR1A = 62377;     // Timer set for 1000 mSec using correction factor
  // 62500 is nominal for 1000 mSec. Decrease variable to increase clock speed

  // Load Fclk depending of AD9850 or AD9851
  if (w0 == 1) fCLK=180000000;    // Enter clock frequency (30 x 6)
  else         fCLK=125000000;    // AD9850

  //Load offset values for 4AFSK
  OffsetFreq[0] = 0;
  OffsetFreq[1] = 1.43*pow(2,32)/fCLK;
  OffsetFreq[2] = 2.93*pow(2,32)/fCLK;
  OffsetFreq[3] = 4.39*pow(2,32)/fCLK;
  
  
  // set up the LCD for 16 columns and 2 rows 
  lcd.begin(16, 2); 
  lcd.clear();

  //Set up GPS pin
  //pinMode(GPSpin, INPUT);
  //digitalWrite(GPSpin, HIGH); // internal pull-up enabled

  // Set up TX pin to output
  pinMode(txPin, OUTPUT);

  // Set up GPS LED pin to output
  pinMode(txLed, OUTPUT);  

  // Set GPS Serial to 9600 baud
  Serial.begin(9600);

  // Set up Calibration, Offset and Fix frequency buttons 
  pinMode(CalOffsetButton, INPUT);      // declare pushbutton as input 
  digitalWrite(CalOffsetButton, HIGH);  // internal pull-up enabled
  pinMode(FixFreqButton, INPUT);        // declare pushbutton as input 
  digitalWrite(FixFreqButton, HIGH);    // internal pull-up enabled
  pinMode(CalOffUpButton, INPUT);       // declare pushbutton as input 
  digitalWrite(CalOffUpButton, HIGH);   // internal pull-up enabled
  pinMode(CalOffDwnButton, INPUT);      // declare pushbutton as input 
  digitalWrite(CalOffDwnButton, HIGH);  // internal pull-up enabled

  
  // Set PB1, PB2, PB3 (= D9 t/m D11) as OUTPUT for Band selection
  DDRB |= (1 << PB1) | (1 << PB2) | (1 << PB3);


// Turn on LCD and startscreen
  lcd.display();
  lcd.setCursor(0,0);
  lcd.print(F(" WSPR Tx PA0RWE "));
  lcd.setCursor(0,1);
  lcd.print(F("  Version: 5.1  "));  
  delay (2000);                       // Wait 2 seconds
  lcd.setCursor(14,0);
  lcd.print(F("  ")); 
  dispok = true;                      // Ready to start display time
  
  // Set up AD985x
  setup_dds();

  // turn off transmitter
  TempWord = 0;
  TransmitSymbol();

  // Display "IDLE" on LCD
  lcd.setCursor(8,1);
  lcd.print(F(" IDLE   "));

  // Load Cal Factor
  CalFactor.b1 = EEPROM.read(50);
  CalFactor.b2 = EEPROM.read(51);
  CalFactor.b3 = EEPROM.read(52);
  CalFactor.b4 = EEPROM.read(53);
  BandOffset   = EEPROM.read(54);

  if ((BandOffset < 0) || (BandOffset > 200)) BandOffset = 100;

  setfreq(); 

  // Begin WSPR message calculation
  encode_call();
  encode_locator();
  encode_conv();
  interleave_sync();

/******************************************************************************
     Calibration process follows after pushing Cal button during startup:
******************************************************************************/
  if(digitalRead(CalOffsetButton) == LOW)
  {
    FreqWord = TenMHz*pow(2,32)/fCLK;
    TempWord = FreqWord;
    TransmitSymbol();
    TIMSK1 = 0;             // Disable timer1 interrupt (second clock)
    TIMSK2 = 0;             // Disable timer2 interrupt (WSPR)

    lcd.setCursor(0,0);
    lcd.print(F("Adjust to 10MHz"));
    lcd.setCursor(0,1);
    lcd.print(F("Cal Factor= "));
    lcd.setCursor(12,1);
    lcd.print(CalFactor.value);
    calibrate();
  }

//  attachInterrupt(digitalPinToInterrupt(CalOffsetButton), set_offset_flag, LOW);
  attachInterrupt(digitalPinToInterrupt(FixFreqButton), set_fixed_flag, LOW);
}

/******************************************************************
 *    L O O P 
 ******************************************************************/
void loop()
{
  if (digitalRead(CalOffsetButton) == LOW) set_offset_freq();
  if (FixFreqFlag) set_fixed_slot();
  
  if (RXflag == 1) {
    digitalWrite (txLed, LOW);
    static int i = 0;
    if (Serial.available())
    {
      digitalWrite (txLed, HIGH); 
      char ch = Serial.read();
      if (ch != '\n' && i < sentenceSize)
      {
        sentence[i] = ch;
        i++;
      }
      else
      {
        sentence[i] = '\0';
        i = 0;
        displayGPS();
      }
    }
  }
}

/******************************************************************
 *    S E T U P   D D S
 ******************************************************************/
void setup_dds()
{
  // DDS pins for data, clock and load
  pinMode (data, OUTPUT);   // DDS pins as output
  pinMode (clock, OUTPUT);  
  pinMode (load, OUTPUT);   

  digitalWrite(data, LOW);  // internal pull-down
  digitalWrite(clock, LOW);
  digitalWrite(load, LOW);
  
// Wait 1 seconds for the AD985x during power up
  delay (1000);
  init_dds();
  reset_dds();
}

void init_dds()
{
  digitalWrite(clock, LOW);
  digitalWrite(load, LOW);
  digitalWrite(data, LOW);
}

void reset_dds()
{
  // Reset sequence is:
  // CLOCK & LOAD = LOW
  // Pulse RESET high for a few uS (use 5 uS here)
  // Pulse CLOCK high for a few uS (use 5 uS here)
  // Set DATA to ZERO and pulse LOAD for a few uS (use 5 uS here)
  
  // data sheet diagrams show only RESET and CLOCK being used to reset the device, but I see no output unless I also
  // toggle the LOAD line here.
  
  digitalWrite(clock, LOW);
  digitalWrite(load, LOW);
    
  digitalWrite(clock, LOW);
  delay(5);
  digitalWrite(clock, HIGH);  //pulse CLOCK
  delay(5);
  digitalWrite(clock, LOW);
  delay(5);
  digitalWrite(data, LOW);    //make sure DATA pin is LOW
     
  digitalWrite(load, LOW);
  delay(5);
  digitalWrite(load, HIGH);   //pulse LOAD
  delay(5);
  digitalWrite(load, LOW);
  // DDS is RESET now
}

/******************************************************************
 *   E N C O D I N G 
 ******************************************************************/
void encode() 
{
  encode_call();
  encode_locator();
  encode_conv();
  interleave_sync();
};

//******************************************************************
// normalize characters 0..9 A..Z Space in order 0..36
char chr_normf(char bc ) 
{
  char cc=36;
  if (bc >= '0' && bc <= '9') cc=bc-'0';
  if (bc >= 'A' && bc <= 'Z') cc=bc-'A'+10;
  if (bc >= 'a' && bc <= 'z') cc=bc-'a'+10;  
  if (bc == ' ' ) cc=36;

  return(cc);
}

//******************************************************************
void encode_call()
{
  unsigned long t1;

  // coding of callsign
  if (chr_normf(call[2]) > 9) 
  {
    call[5] = call[4];
    call[4] = call[3]; 
    call[3] = call[2];
    call[2] = call[1];
    call[1] = call[0];
    call[0] = ' ';
  }

  n1=chr_normf(call[0]);
  n1=n1*36+chr_normf(call[1]);
  n1=n1*10+chr_normf(call[2]);
  n1=n1*27+chr_normf(call[3])-10;
  n1=n1*27+chr_normf(call[4])-10;
  n1=n1*27+chr_normf(call[5])-10;

  // merge coded callsign into message array c[]
  t1=n1;
  c[0]= t1 >> 20;
  t1=n1;
  c[1]= t1 >> 12;
  t1=n1;
  c[2]= t1 >> 4;
  t1=n1;
  c[3]= t1 << 4;
}

//******************************************************************
void encode_locator()
{
  unsigned long t1;
  // coding of locator
  m1=179-10*(chr_normf(locator[0])-10)-chr_normf(locator[2]);
  m1=m1*180+10*(chr_normf(locator[1])-10)+chr_normf(locator[3]);
  m1=m1*128+power+64;

  // merge coded locator and power into message array c[]
  t1=m1;
  c[3]= c[3] + ( 0x0f & t1 >> 18);
  t1=m1;
  c[4]= t1 >> 10;
  t1=m1;
  c[5]= t1 >> 2;
  t1=m1;
  c[6]= t1 << 6;
}

//******************************************************************
// convolutional encoding of message array c[] into a 162 bit stream
void encode_conv()
{
  int bc=0;
  int cnt=0;
  int cc;
  unsigned long sh1=0;

  cc=c[0];

  for (int i=0; i < 81;i++) {
    if (i % 8 == 0 ) {
      cc=c[bc];
      bc++;
    }
    if (cc & 0x80) sh1=sh1 | 1;

    symt[cnt++]=parity(sh1 & 0xF2D05351);
    symt[cnt++]=parity(sh1 & 0xE4613C47);

    cc=cc << 1;
    sh1=sh1 << 1;
  }
}

//******************************************************************
byte parity(unsigned long li)
{
  byte po = 0;
  while(li != 0)
  {
    po++;
    li&= (li-1);
  }
  return (po & 1);
}

//******************************************************************
// interleave reorder the 162 data bits and and merge table with the sync vector
void interleave_sync()
{
  int ii,ij,b2,bis,ip;
  ip=0;

  for (ii=0;ii<=255;ii++) {
    bis=1;
    ij=0;
    for (b2=0;b2 < 8 ;b2++) {
      if (ii & bis) ij= ij | (0x80 >> b2);
      bis=bis << 1;
    }
    if (ij < 162 ) {
      sym[ij]= SyncVec[ij] +2*symt[ip];
      ip++;
    }
  }
}

/******************************************************************
 *  S E T   A N D   D I S P L A Y   F R E Q 
 *  Determine time slot and load band frequency data.  
 *  Display frequency and calculate frequency word for DDS
 ******************************************************************/
void setfreq()
{
  char buf[10];
  lcd.setCursor(0,0);

  ltoa(band[timeslot]+BandOffset,buf,10);  // Convert long to ASCII

  if (buf[7]==0) {          // < 10 MHz
    lcd.print(buf[0]);
    lcd.print(',');
    lcd.print(buf[1]);
    lcd.print(buf[2]);
    lcd.print(buf[3]);
    lcd.print('.');
    lcd.print(buf[4]);
    lcd.print(buf[5]);
    lcd.print(buf[6]);
    lcd.print(" KHz ");
  }
  else {                    // >= 10MHz
    lcd.print(buf[0]);
    lcd.print(buf[1]);
    lcd.print(',');
    lcd.print(buf[2]);
    lcd.print(buf[3]);
    lcd.print(buf[4]);
    lcd.print('.');
    lcd.print(buf[5]);
    lcd.print(buf[6]);
    lcd.print(buf[7]);
    lcd.print(" KHz");
  }

  FreqWord = ((band[timeslot]+BandOffset)+(CalFactor.value*((band[timeslot]+BandOffset)/pow(10,7))))*pow(2,32)/fCLK;

// ******************** Update band select ********************************************
  
  // Clear bits PB1-PB3
  PORTB &= ~((1 << PB1) | (1 << PB2) | (1 << PB3));

  // Set bits according bandSelect[timeslot]
  PORTB |= ((bandSelect[timeslot] & 0x07) << PB1); // 0x07 = mask voor 3 bits

  return;
}

/******************************************************************
 *  T R A N S M I T
 *  Determine if it is time to transmit. If so, determine if it is
 *  time to transmit the WSPR message
 ******************************************************************/
void transmit()
{
  if(GPSinhibitFlag == 1)return;
  else  
    if (TransmitFlag[timeslot] == 1)
    { // Start WSPR transmit process
      RXflag = 0;                   // Disable GPS receiver
      lcd.setCursor(9,1); 
      lcd.print(Band[timeslot]);
      lcd.print(" TX ");
      digitalWrite(txPin,HIGH);     // Transmit control ON
      digitalWrite(txLed,HIGH);
      bb =0;
      count = 0;                    // Start WSPR symbol transmit process
      TIMSK2 = 2;                   // Enable timer2 interrupt 
      return;
    }
    else
    { // Turn off transmitter and idle
      TIMSK2 = 0;                   // Turn off WSPR timer
      lcd.setCursor(8,1);
      lcd.print(F(" IDLE   "));
      digitalWrite(txPin,LOW);      // Transmit control OFF
      digitalWrite(txLed,LOW);
      RXflag = 1;                   // Enable GPS receiver
      TempWord=0;
      TransmitSymbol();             // Set DDS frequency to 0 Hz
      return;
    }
}  

/******************************************************************
 *   T U N E    D D S (Serial mode)
 ******************************************************************/
void TransmitSymbol()
{
  digitalWrite (load, LOW); // take load pin low

  for(int i = 0; i < 32; i++)
  {
    if ((TempWord & 1) == 1)
      outOne();
    else
      outZero();
    TempWord = TempWord >> 1;
  }

// Setup first byte
  if (w0 == 1)  byte_out(0x09); // AD9851
  else          byte_out(0x00); // AD9850

  digitalWrite (load, HIGH); // Take load pin high again
}

void byte_out(unsigned char byte)
{
  int i;

  for (i = 0; i < 8; i++)
  {
    if ((byte & 1) == 1)
      outOne();
    else
      outZero();
    byte = byte >> 1;
  }
}

void outOne()
{
  digitalWrite(data, HIGH);
  digitalWrite(clock, HIGH);
  delayMicroseconds(1);
  digitalWrite(clock, LOW);
}

void outZero()
{
  digitalWrite(data, LOW);
  digitalWrite(clock, HIGH);
  delayMicroseconds(1);
  digitalWrite(clock, LOW);
}


/******************************************************************
 *   G P S   T I M I N G  A N D   S A T S
 ******************************************************************/
void displayGPS()
{
  getField ();
  if (strcmp(words[0], HEADER_RMC) == 0)
  {
    processRMCstring();   // Get time
  }
  else if (strcmp(words[0], HEADER_GGA) == 0)
  {
    processGGAstring ();  // Get # sats
  }
}

void getField () // split string was probably a better name
{
  char* p_input = &sentence[0];
  for (int i=0; i<max_nr_words; ++i)
  {
    words[i] = strsep(&p_input, ",");
  }
}

void processRMCstring ()
{
  time = atof(words [RMC_TIME]);
  if (time >= 0 && time < 240000)           // To prevent rubbish....
  {
    hour = (int)(time * 0.0001);
    minute = int((time * 0.01) - (hour * 100));
    second = int(time - (hour * 10000) - (minute * 100));
  }
}

void processGGAstring ()
{
  gps_sats = atoi(words[GGA_SATS]);
}


/******************************************************************
 *  C A L I B R A T E  
 *  Process to determine frequency calibration factor @ 10 MHz
 ******************************************************************/
void calibrate()
{
  while(digitalRead(CalOffsetButton) == LOW)
  {
    if (digitalRead(CalOffUpButton) == LOW)
    {
      delay (500);
      CalFactor.value = CalFactor.value + 5;
    };
    if (digitalRead(CalOffDwnButton) == LOW)
    {
      delay (500);
      CalFactor.value = CalFactor.value - 5;
    };
    
    TempWord = (TenMHz+CalFactor.value)*pow(2,32)/fCLK;
    TransmitSymbol();
    lcd.setCursor(12,1);
    lcd.print(CalFactor.value);
    lcd.print("   ");
  }

  // Writes CalFactor to address 50 + 3 bytes of EEprom
  EEPROM.write(50,CalFactor.b1);
  EEPROM.write(51,CalFactor.b2);
  EEPROM.write(52,CalFactor.b3);
  EEPROM.write(53,CalFactor.b4);
  
  lcd.setCursor(0,1);
  lcd.print(F("         IDLE   "));
  TempWord = 0;                             // turn off 10 MHz calibrate signal
  TransmitSymbol();
  setfreq(); 
  TIMSK1 = 2;                               // Enable Timer1 Interrupt 
}   

/*********************************************************************************
 *  S E T  F I X E D   F R E Q U E N C Y 
 *  Process to set a fixed Timeslot. This is an ISR called by the FixFreqButton.
 *  If interrupts come faster than 200ms, assume it's a bounce and ignore
 ********************************************************************************/
void set_fixed_flag()               // Sart by interrupt
{
  detachInterrupt(digitalPinToInterrupt(FixFreqButton));    // Prevent firing again

  static unsigned long last_interrupt_time = 0;
  unsigned long interrupt_time = millis();

  if (interrupt_time - last_interrupt_time > 200) 
  {
    if (FixonFlag == 1)            // Fix freq is on
    {
      FixonFlag = 0;                // Turn fix freq off
      lcd.setCursor(14,0);          // Fixed Indicator off
      lcd.print("  ");
      FixFreqFlag = 1;
      attachInterrupt(digitalPinToInterrupt(FixFreqButton), set_fixed_flag, LOW);      
    }
    else FixFreqFlag = 1;              // Turn fix freq on
  }
  last_interrupt_time = interrupt_time;
}

//**************************************************************************************
void set_fixed_slot()
{
  NoDispFlag = 1;                                           // Stop Time display
  FixonFlag = 0;
  digitalWrite(txPin,LOW);                                  // Transmit control OFF
  digitalWrite(txLed,LOW);
  
  lcd.setCursor(0,0);
  lcd.print(F(" Set Fixed Slot "));
  lcd.setCursor(0,1);
  lcd.print(F("Timeslot = "));
  lcd.print(timeslot);
  lcd.print(F("   "));
  
  while(digitalRead(FixFreqButton) == LOW)                  // Do Timeslot until button is released
  {
    if (digitalRead(CalOffUpButton) == LOW)                 // UP button pressed
    {
      delay(500);
      fixedslot = fixedslot + 1;
      if (fixedslot > 9) fixedslot = 9;
      FixonFlag = 1;
    };
    
    if (digitalRead(CalOffDwnButton) == LOW)
    {
      delay(500);                                           // Debounce delay
      fixedslot = fixedslot - 1;
      if (fixedslot < 0) fixedslot = 0;   
      FixonFlag = 1;   
    };
    lcd.setCursor(9,1);
    lcd.print(fixedslot);
    lcd.print(F("   "));
  }

  if (FixonFlag == 1)
  {
    lcd.setCursor(14,0);                                    // Fixed Indicator
    lcd.print(" F");
    timeslot = fixedslot;
  }

  Serial.println(fixedslot);
  lcd.setCursor(0,1);
  lcd.print(F("         IDLE   "));
  FreqWord = ((band[timeslot]+BandOffset)+(CalFactor.value*((band[timeslot]+BandOffset)/pow(10,7))))*pow(2,32)/fCLK;
  setfreq();
  FixFreqFlag = 0;
  NoDispFlag = 0;
  attachInterrupt(digitalPinToInterrupt(FixFreqButton), set_fixed_flag, LOW);
}


/**************************************************************************************
 *   S E T   B A N D O F F S E T   I N   2 5 H z   S T E P S
 **************************************************************************************/
void set_offset_freq()
{
  NoDispFlag = 1;                                           // Stop Time display
  digitalWrite(txPin,LOW);                                  // Transmit control OFF
  digitalWrite(txLed,LOW);
  
  lcd.setCursor(0,0);
  lcd.print(F("Set Freq Offset "));
  lcd.setCursor(0,1);
  lcd.print(F("Offset = "));
  lcd.print(BandOffset);
  lcd.print(F("   "));
  
  while(digitalRead(CalOffsetButton) == LOW)                // Do offset until button is released
  {
    if (digitalRead(CalOffUpButton) == LOW)                 // UP button pressed
    {
      delay(500);
      BandOffset = BandOffset + 25;
      if (BandOffset > 200) BandOffset = 100;
    }
    
    if (digitalRead(CalOffDwnButton) == LOW)
    {
      delay(500);
      BandOffset = BandOffset - 25;
      if (BandOffset < 0) BandOffset = 100;      
    };
    lcd.setCursor(9,1);
    lcd.print(BandOffset);
    lcd.print(F("   "));
  }
  
  EEPROM.write(54,BandOffset);                              // Store Offset
  lcd.setCursor(0,1);
  lcd.print(F("         IDLE   "));
  FreqWord = ((band[timeslot]+BandOffset)+(CalFactor.value*((band[timeslot]+BandOffset)/pow(10,7))))*pow(2,32)/fCLK;
  setfreq();
  NoDispFlag = 0;
}

      
/******************************************************************
 *   D I S P L A Y   T I M E 
 ******************************************************************/
void displaytime()
{
  // Minute to transmit in Fixed mode (every 6 minutes)
  int Txmin[] = {0,1,6,7,12,13,18,19,24,25,30,31,36,37,42,43,48,49,54,55}; 
  TxOnFlag = 1;
  
  if (dispok); {                            // Display after intro
    lcd.setCursor(0,1);
    if (hour < 10) lcd.print ("0");
    lcd.print (hour);
    lcd.print (":");
    if (minute < 10) lcd.print ("0");
    lcd.print (minute);
    lcd.print (":");
    if (second < 10) lcd.print ("0");
    lcd.print (second);

    if ((gps_sats > 0) && (gps_sats < 15) && (digitalRead(txLed) == LOW))
    {
      lcd.print ("  Sat:"); 
      lcd.print(gps_sats);
    }

    if (FixonFlag == 1) {
      if (arrayIncludeElement(Txmin, minute)){
      TxOnFlag = 1;
      } else {
      TxOnFlag = 0;
      }
    }  
  }
}


boolean arrayIncludeElement(int array[], int element) {
  for (int i = 0; i < 20; i++) {
    if (array[i] == element) {
      return true;
    }
  }
  return false;
}
// End of file
